//------------------------------------------------------------------------------
// PCILeech FPGA
// PCIe configuration module - CFG handling for Artix-7.
//
// (c) Ulf Frisk, 2018-2024
//
// 说明：
// 1) 本模块负责：
//    - FT601(系统时钟域 clk_sys) -> PCIe(时钟域 clk_pcie) 的 CFG 命令 FIFO 传输
//    - PCIe core 的 cfg_mgmt 读写控制与结果回传
//    - 将 CFG/状态信息映射到 ro/rw 寄存器文件
//    - 你新增的 MSI-X(MWR64) TLP 触发逻辑：o_int + i_valid 时注入一个静态 TLP
//
// 2) ro/rw 寄存器文件：
//    - ro: 只读寄存器视图(用于上位机读取状态/核心信号)
//    - rw: 可写寄存器视图(用于上位机控制 cfg_mgmt、链路控制、中断等)
//
// 3) base_address_register_reg：
//    - 通过 cfg_mgmt 读 BAR0 保存到寄存器并输出给上层模块使用
//------------------------------------------------------------------------------


`timescale 1ns / 1ps
`include "pcileech_header.svh"

module pcileech_pcie_cfg_a7(
    input                   rst,
    input                   clk_sys,        // 系统时钟域(FT601/USB侧)
    input                   clk_pcie,       // PCIe 核心时钟域
    IfPCIeFifoCfg.mp_pcie   dfifo,          // FT601 <-> FPGA FIFO 接口(封装)
    IfPCIeSignals.mpm       ctx,            // PCIe Core CFG/状态信号集合
    IfAXIS128.source        tlps_static,    // 128-bit AXIS 静态 TLP 注入通道

    // 你的外部中断/写入输入(用于生成 MSI-X TLP)
    input                   i_valid,
    input [31:0]            i_addr,
    input [31:0]            i_data,
    input                   int_enable,     // 允许定时触发 o_int
    output [15:0]           pcie_id,        // {bus,dev,func} 组合(来自 ro)
    output wire [31:0]      base_address_register
);

    //==========================================================================
    // 1) TickCount64：PCIe 时钟域下的计数器(用于节流/定时)
    //==========================================================================
    time tickcount64 = 0;
    always @ (posedge clk_pcie)
        tickcount64 <= tickcount64 + 1;


    //==========================================================================
    // 2) FT601 -> PCIe：CFG 命令跨时钟域 FIFO
    //    - dfifo.tx_* 在 clk_sys 写入
    //    - clk_pcie 侧读取为 in_dout
    //    FIFO 深度：512 / 64-bit
    //==========================================================================
    reg             in_rden;     // PCIe 域读使能
    wire [63:0]     in_dout;     // PCIe 域读出的 64-bit 命令/数据
    wire            in_empty;
    wire            in_valid;    // FIFO 输出有效

    // 命令字段拆解(按原工程约定)
    reg [63:0]      in_data64;
    wire [31:0]     in_data32   = in_data64[63:32];
    wire [15:0]     in_data16   = in_data64[31:16];
    wire [3:0]      in_type     = in_data64[15:12];

    fifo_64_64 i_fifo_pcie_cfg_tx(
        .rst            ( rst            ),
        .wr_clk         ( clk_sys        ),
        .rd_clk         ( clk_pcie       ),
        .din            ( dfifo.tx_data  ),
        .wr_en          ( dfifo.tx_valid ),
        .rd_en          ( in_rden        ),
        .dout           ( in_dout        ),
        .full           (                ),
        .empty          ( in_empty       ),
        .valid          ( in_valid       )
    );


    //==========================================================================
    // 3) PCIe -> FT601：CFG 结果/回包跨时钟域 FIFO
    //    - out_data/out_wren 在 clk_pcie 写入
    //    - clk_sys 侧由 dfifo.rx_rd_en 读取
    //    FIFO 深度：512 / 32-bit
    //==========================================================================
    reg             out_wren;    // PCIe 域写使能
    reg [31:0]      out_data;    // PCIe 域写入的 32-bit 数据
    wire            pcie_cfg_rx_almost_full;

    fifo_32_32_clk2 i_fifo_pcie_cfg_rx(
        .rst            ( rst                     ),
        .wr_clk         ( clk_pcie                ),
        .rd_clk         ( clk_sys                 ),
        .din            ( out_data                ),
        .wr_en          ( out_wren                ),
        .rd_en          ( dfifo.rx_rd_en          ),
        .dout           ( dfifo.rx_data           ),
        .full           (                         ),
        .almost_full    ( pcie_cfg_rx_almost_full ),
        .empty          (                         ),
        .valid          ( dfifo.rx_valid          )
    );


    //==========================================================================
    // 4) 寄存器文件：ro(只读) / rw(可读写)
    //    ro：主要映射 ctx 的状态信号给上位机读
    //    rw：上位机写入后，驱动 ctx 的 cfg_mgmt / 中断 / 链路控制等
    //==========================================================================
    wire    [383:0] ro;
    reg     [703:0] rw;

    // 一些内部专用寄存器(不对上位机直接暴露为“字段”，但仍可映射到 ro)
    reg             rwi_cfg_mgmt_rd_en;      // 内部触发 cfg_mgmt 读
    reg             rwi_cfg_mgmt_wr_en;      // 内部触发 cfg_mgmt 写
    reg             rwi_cfgrd_valid;         // 最近一次 cfg_mgmt 读写完成标志
    reg     [9:0]   rwi_cfgrd_addr;          // 最近访问 dwaddr
    reg     [3:0]   rwi_cfgrd_byte_en;       // 最近使用的 byte_en
    reg     [31:0]  rwi_cfgrd_data;          // 最近一次读取到的数据(或写回读到的 do)
    reg     [31:0]  rwi_count_cfgspace_status_cl; // 状态寄存器自动清计数器
    bit     [31:0]  base_address_register_reg;    // 保存 BAR0 读取结果

    // 输出 BAR(给外部模块使用)
    assign base_address_register = base_address_register_reg;


    //==========================================================================
    // 5) ro 寄存器映射(只读视图)
    //    注意：ro 是 bit-level 拼接映射，上位机按地址读取 16-bit 片段
    //==========================================================================
    // +000: MAGIC
    assign ro[15:0]     = 16'h2301;

    // +002: SPECIAL(反映 PCIe core cfg_mgmt 的实际使能)
    assign ro[16]       = ctx.cfg_mgmt_rd_en;
    assign ro[17]       = ctx.cfg_mgmt_wr_en;
    assign ro[31:18]    = 0;

    // +004: BYTECOUNT(以字节为单位，little-endian 的表示方式由上位机处理)
    assign ro[63:32]    = $bits(ro) >> 3;

    // +008: PCIe BDF 状态
    assign ro[71:64]    = ctx.cfg_bus_number;
    assign ro[76:72]    = ctx.cfg_device_number;
    assign ro[79:77]    = ctx.cfg_function_number;

    // +00A ~ +00D: PCIe PL PHY 状态
    assign ro[85:80]    = ctx.pl_ltssm_state;
    assign ro[87:86]    = ctx.pl_rx_pm_state;
    assign ro[90:88]    = ctx.pl_tx_pm_state;
    assign ro[93:91]    = ctx.pl_initial_link_width;
    assign ro[95:94]    = ctx.pl_lane_reversal_mode;
    assign ro[97:96]    = ctx.pl_sel_lnk_width;
    assign ro[98]       = ctx.pl_phy_lnk_up;
    assign ro[99]       = ctx.pl_link_gen2_cap;
    assign ro[100]      = ctx.pl_link_partner_gen2_supported;
    assign ro[101]      = ctx.pl_link_upcfg_cap;
    assign ro[102]      = ctx.pl_sel_lnk_rate;
    assign ro[103]      = ctx.pl_directed_change_done;
    assign ro[104]      = ctx.pl_received_hot_rst;
    assign ro[126:105]  = 0;
    assign ro[127]      = ctx.cfg_mgmt_rd_wr_done;

    // +010: cfg_mgmt_do(最近一次 cfg_mgmt 返回的数据)
    assign ro[159:128]  = ctx.cfg_mgmt_do;

    // +014 ~ : 常见 cfg/status
    assign ro[175:160]  = ctx.cfg_command;
    assign ro[176]      = ctx.cfg_aer_rooterr_corr_err_received;
    assign ro[177]      = ctx.cfg_aer_rooterr_corr_err_reporting_en;
    assign ro[178]      = ctx.cfg_aer_rooterr_fatal_err_received;
    assign ro[179]      = ctx.cfg_aer_rooterr_fatal_err_reporting_en;
    assign ro[180]      = ctx.cfg_aer_rooterr_non_fatal_err_received;
    assign ro[181]      = ctx.cfg_aer_rooterr_non_fatal_err_reporting_en;
    assign ro[182]      = ctx.cfg_bridge_serr_en;
    assign ro[183]      = ctx.cfg_received_func_lvl_rst;
    assign ro[186:184]  = ctx.cfg_pcie_link_state;
    assign ro[187]      = ctx.cfg_pmcsr_pme_en;
    assign ro[189:188]  = ctx.cfg_pmcsr_powerstate;
    assign ro[190]      = ctx.cfg_pmcsr_pme_status;
    assign ro[191]      = 0;

    assign ro[207:192]  = ctx.cfg_dcommand;
    assign ro[223:208]  = ctx.cfg_dcommand2;
    assign ro[239:224]  = ctx.cfg_dstatus;
    assign ro[255:240]  = ctx.cfg_lcommand;
    assign ro[271:256]  = ctx.cfg_lstatus;
    assign ro[287:272]  = ctx.cfg_status;

    // +024: TX buffer availability / cfg req
    assign ro[293:288]  = ctx.tx_buf_av;
    assign ro[294]      = ctx.tx_cfg_req;
    assign ro[295]      = ctx.tx_err_drop;
    assign ro[302:296]  = ctx.cfg_vc_tcvc_map;
    assign ro[303]      = 0;

    // +026: Root control / to_turnoff
    assign ro[304]      = ctx.cfg_root_control_pme_int_en;
    assign ro[305]      = ctx.cfg_root_control_syserr_corr_err_en;
    assign ro[306]      = ctx.cfg_root_control_syserr_fatal_err_en;
    assign ro[307]      = ctx.cfg_root_control_syserr_non_fatal_err_en;
    assign ro[308]      = ctx.cfg_slot_control_electromech_il_ctl_pulse;
    assign ro[309]      = ctx.cfg_to_turnoff;
    assign ro[319:310]  = 0;

    // +028: Interrupt 相关状态(来自 core)
    assign ro[327:320]  = ctx.cfg_interrupt_do;
    assign ro[330:328]  = ctx.cfg_interrupt_mmenable;
    assign ro[331]      = ctx.cfg_interrupt_msienable;
    assign ro[332]      = ctx.cfg_interrupt_msixenable;
    assign ro[333]      = ctx.cfg_interrupt_msixfm;
    assign ro[334]      = ctx.cfg_interrupt_rdy;
    assign ro[335]      = 0;

    // +02A ~ +02C: 最近一次 cfgspace 访问结果
    assign ro[345:336]  = rwi_cfgrd_addr;
    assign ro[346]      = 0;
    assign ro[347]      = rwi_cfgrd_valid;
    assign ro[351:348]  = rwi_cfgrd_byte_en;
    assign ro[383:352]  = rwi_cfgrd_data;


    //==========================================================================
    // 6) rw 寄存器位定义(仅列出你用到的控制位)
    //==========================================================================
    localparam integer RWPOS_CFG_RD_EN                 = 16; // 写 1 触发 cfg_mgmt 读
    localparam integer RWPOS_CFG_WR_EN                 = 17; // 写 1 触发 cfg_mgmt 写
    localparam integer RWPOS_CFG_WAIT_COMPLETE         = 18; // 等待 cfg_mgmt 完成期间是否阻塞新命令
    localparam integer RWPOS_CFG_STATIC_TLP_TX_EN      = 19; // 原静态 TLP 发送开关(你这里改成 msix_* 逻辑了)
    localparam integer RWPOS_CFG_CFGSPACE_STATUS_CL_EN = 20; // 自动清 status(如 master abort)使能
    localparam integer RWPOS_CFG_CFGSPACE_COMMAND_EN   = 21; // 自动设置 command(bus master 等)使能


    //==========================================================================
    // 7) 初始化任务：复位/上电默认值
    //==========================================================================
    task pcileech_pcie_cfg_a7_initialvalues;
        begin
            out_wren                 <= 1'b0;
            rwi_cfg_mgmt_rd_en       <= 1'b0;
            rwi_cfg_mgmt_wr_en       <= 1'b0;
            base_address_register_reg<= 32'h00000000;

            // +000: MAGIC(可写侧 magic，供上位机识别 RW 映射存在)
            rw[15:0]     <= 16'h6745;

            // +002: 控制位(写 1 触发动作)
            rw[16]       <= 0;        // CFG RD EN
            rw[17]       <= 0;        // CFG WR EN
            rw[18]       <= 0;        // WAIT COMPLETE
            rw[19]       <= 0;        // STATIC TLP TX EN
            rw[20]       <= 1;        // STATUS AUTO CLEAR
            rw[21]       <= 0;        // COMMAND AUTO SET
            rw[31:22]    <= 0;

            // +004: bytecount
            rw[63:32]    <= $bits(rw) >> 3;

            // +008: DSN(设备序列号)
            rw[127:64]   <= 64'h0010_0020_3EF5_40;

            // +010: cfg_mgmt 输入参数
            rw[159:128]  <= 0;        // cfg_mgmt_di
            rw[169:160]  <= 0;        // cfg_mgmt_dwaddr
            rw[170]      <= 0;        // wr_readonly
            rw[171]      <= 0;        // wr_rw1c_as_rw
            rw[175:172]  <= 4'hf;     // byte_en 默认全开

            // +016~+017: 链路控制
            rw[176]      <= 0;        // pl_directed_link_auton
            rw[178:177]  <= 0;        // pl_directed_link_change
            rw[179]      <= 1;        // pl_directed_link_speed
            rw[181:180]  <= 0;        // pl_directed_link_width
            rw[182]      <= 1;        // pl_upstream_prefer_deemph
            rw[183]      <= 0;        // pl_transmit_hot_rst
            rw[184]      <= 0;        // pl_downstream_deemph_source
            rw[191:185]  <= 0;

            // +018~+019: 中断控制(原工程 rw 映射，此处你改成 cfg_* 寄存器驱动)
            rw[199:192]  <= 0;        // cfg_interrupt_di
            rw[204:200]  <= 0;        // msgnum
            rw[205]      <= 0;        // assert
            rw[206]      <= 0;        // interrupt
            rw[207]      <= 0;        // stat

            // +01A~+01B: PM / turnoff / NP
            rw[209:208]  <= 0;
            rw[210]      <= 0;
            rw[211]      <= 0;
            rw[212]      <= 0;
            rw[213]      <= 0;
            rw[214]      <= 0;
            rw[215]      <= 0;
            rw[216]      <= 0;        // cfg_turnoff_ok
            rw[217]      <= 1;        // rx_np_ok
            rw[218]      <= 1;        // rx_np_req
            rw[219]      <= 1;        // tx_cfg_gnt
            rw[223:220]  <= 0;

            // +01C~+050: 原静态 TLP 缓冲/控制(你目前不使用原始发送状态机了)
            rw[224+:8]    <= 0;
            rw[232+:8]    <= 0;
            rw[240+:16]   <= 0;
            rw[256+:384]  <= 0;
            rw[640+:32]   <= 0;

            // +054: status 清理定时器(默认 1ms @ 62.5MHz -> 62500 ticks)
            rw[672+:32]   <= 62500;
        end
    endtask


    //==========================================================================
    // 8) ctx 信号连接：rw -> PCIe core 控制信号
    //    注意：cfg_mgmt_rd_en/wr_en 需要握手 done 才能拉低
    //==========================================================================
    assign ctx.cfg_mgmt_rd_en         = rwi_cfg_mgmt_rd_en & ~ctx.cfg_mgmt_rd_wr_done;
    assign ctx.cfg_mgmt_wr_en         = rwi_cfg_mgmt_wr_en & ~ctx.cfg_mgmt_rd_wr_done;

    assign ctx.cfg_dsn                = rw[127:64];
    assign ctx.cfg_mgmt_di            = rw[159:128];
    assign ctx.cfg_mgmt_dwaddr        = rw[169:160];
    assign ctx.cfg_mgmt_wr_readonly   = rw[170];
    assign ctx.cfg_mgmt_wr_rw1c_as_rw = rw[171];
    assign ctx.cfg_mgmt_byte_en       = rw[175:172];

    assign ctx.pl_directed_link_auton      = rw[176];
    assign ctx.pl_directed_link_change     = rw[178:177];
    assign ctx.pl_directed_link_speed      = rw[179];
    assign ctx.pl_directed_link_width      = rw[181:180];
    assign ctx.pl_upstream_prefer_deemph   = rw[182];
    assign ctx.pl_transmit_hot_rst         = rw[183];
    assign ctx.pl_downstream_deemph_source = rw[184];

    // 你这里将中断相关改为寄存器驱动(替代原 rw 映射直连)
    reg [7:0] cfg_int_di;
    reg [4:0] cfg_msg_num;
    reg       cfg_int_assert;
    reg       cfg_int_valid;
    wire      cfg_int_ready = ctx.cfg_interrupt_rdy;
    reg       cfg_int_stat;

    assign ctx.cfg_interrupt_di             = cfg_int_di;
    assign ctx.cfg_pciecap_interrupt_msgnum = cfg_msg_num;
    assign ctx.cfg_interrupt_assert         = cfg_int_assert;
    assign ctx.cfg_interrupt                = cfg_int_valid;
    assign ctx.cfg_interrupt_stat           = cfg_int_stat;

    assign ctx.cfg_pm_force_state      = rw[209:208];
    assign ctx.cfg_pm_force_state_en   = rw[210];
    assign ctx.cfg_pm_halt_aspm_l0s    = rw[211];
    assign ctx.cfg_pm_halt_aspm_l1     = rw[212];
    assign ctx.cfg_pm_send_pme_to      = rw[213];
    assign ctx.cfg_pm_wake             = rw[214];
    assign ctx.cfg_trn_pending         = rw[215];
    assign ctx.cfg_turnoff_ok          = rw[216];
    assign ctx.rx_np_ok                = rw[217];
    assign ctx.rx_np_req               = rw[218];
    assign ctx.tx_cfg_gnt              = rw[219];

    // 输出 pcie_id = {bus,dev,func}
    assign pcie_id = ro[79:64];


    //==========================================================================
    // 9) 你新增的 MSI-X “静态 TLP 注入通道”实现
    //    - 通过 tlps_static 发送一个 MWR64 TLP(4DW header + 1DW data)
    //    - 这里的打包方式：{DW4(data), DW3(addr low), DW2(reqID..), DW1(header)}
    //
    //    注意：这里的 tlps_static.tkeepdw 固定 4'hf，且 tlast 固定 1
    //          表示每次只打一拍 128-bit(4 dwords)，你的 DW4 实际塞进最高 32
    //          如果你的核心/下游 expects has_data 表示还有数据，请确认协议一致。
    //==========================================================================
    bit         msix_valid;       // AXIS tvalid
    bit         msix_has_data;    // 自定义侧带：表示“这次 TLP 有数据”(你用于配合 ready)
    bit [127:0] msix_tlp;         // 128-bit TLP payload

    assign tlps_static.tdata[127:0] = msix_tlp;
    assign tlps_static.tkeepdw      = 4'hf;
    assign tlps_static.tlast        = 1'b1;
    assign tlps_static.tuser[0]     = 1'b1;
    assign tlps_static.tvalid       = msix_valid;
    assign tlps_static.has_data     = msix_has_data;


    //==========================================================================
    // 10) CFG 命令解码：从 in_dout 中提取地址/掩码/数据
    //==========================================================================
    integer i_write;

    // 约定：in_dout[31:16] 为“字节地址”
    wire [15:0] in_cmd_address_byte = in_dout[31:16];

    // bit 地址 = byte_addr[14:0] * 8，低 3bit = 0
    wire [17:0] in_cmd_address_bit  = {in_cmd_address_byte[14:0], 3'b000};

    // 约定：value/mask 为 byte swap 组合
    wire [15:0] in_cmd_value        = {in_dout[48+:8], in_dout[56+:8]};
    wire [15:0] in_cmd_mask         = {in_dout[32+:8], in_dout[40+:8]};

    // 最高位=1 表示访问 rw 区，否则访问 ro 区
    wire        f_rw                = in_cmd_address_byte[15];

    // 读出的 16bit 数据(越界则返回 0)
    wire [15:0] in_cmd_data_in      =
        (in_cmd_address_bit < (f_rw ? $bits(rw) : $bits(ro))) ?
            (f_rw ? rw[in_cmd_address_bit+:16] : ro[in_cmd_address_bit+:16]) :
            16'h0000;

    // 约定：in_dout[12]=read，in_dout[13]=write
    wire        in_cmd_read         = in_dout[12] & in_valid;
    wire        in_cmd_write        = in_dout[13] & in_cmd_address_byte[15] & in_valid;

    // 是否当前存在 cfg_mgmt 操作(或准备触发)
    wire        pcie_cfg_rw_en      =
        rwi_cfg_mgmt_rd_en | rwi_cfg_mgmt_wr_en |
        rw[RWPOS_CFG_RD_EN] | rw[RWPOS_CFG_WR_EN];

    // in_rden：从输入 FIFO 取命令的节流条件：
    // 1) tickcount64[1]：每两拍取一次(留出处理动作时间)
    // 2) 输出 FIFO 不接近满
    // 3) 若 WAIT_COMPLETE=1，则必须等 cfg_mgmt 空闲才取新命令
    assign in_rden =
        tickcount64[1] &
        ~pcie_cfg_rx_almost_full &
        ( ~rw[RWPOS_CFG_WAIT_COMPLETE] | ~pcie_cfg_rw_en );


    //==========================================================================
    // 11) 你用于 MSI-X 的 MWR64 TLP 组包字段
    //==========================================================================
    wire [31:0] HDR_MEMWR64 = 32'b01000000_00000000_00000000_00000001;

    // DW2：Requester ID + Tag/LastBE/FirstBE(你这里固定 0x0000 + 0x0F)
    wire [31:0] MWR64_DW2  = {`_bs16(pcie_id), 8'b00000000, 8'b00001111};

    // DW3：地址低 32(按 dword 对齐)
    wire [31:0] MWR64_DW3  = {i_addr[31:2], 2'b0};

    // DW4：数据
    wire [31:0] MWR64_DW4  = i_data;


    //==========================================================================
    // 12) 上电初始化
    //==========================================================================
    initial pcileech_pcie_cfg_a7_initialvalues();


    //==========================================================================
    // 13) 主状态/逻辑(always)：处理 ro/rw 读写、cfg_mgmt 触发、自动清状态、BAR0读取
    //==========================================================================
    always @ (posedge clk_pcie) begin
        if (rst) begin
            pcileech_pcie_cfg_a7_initialvalues();
        end else begin
            //--------------------------------------------------------------------------
            // 13.1 读：上位机请求读取 ro/rw 时，回包写入 out FIFO
            //--------------------------------------------------------------------------
            out_wren <= in_cmd_read;
            if (in_cmd_read) begin
                // out_data 格式：高 16 = 地址，低 16 = 数据(做一次 byte swap 保持原协议)
                out_data[31:16] <= in_cmd_address_byte;
                out_data[15:0]  <= {in_cmd_data_in[7:0], in_cmd_data_in[15:8]};
            end

            //--------------------------------------------------------------------------
            // 13.2 写：上位机写入 rw 区(按 bit mask 更新)
            //--------------------------------------------------------------------------
            if (in_cmd_write) begin
                for (i_write = 0; i_write < 16; i_write = i_write + 1) begin
                    if (in_cmd_mask[i_write]) begin
                        rw[in_cmd_address_bit+i_write] <= in_cmd_value[i_write];
                    end
                end
            end

            //--------------------------------------------------------------------------
            // 13.3 状态寄存器自动清除/命令寄存器自动设置
            //     - 定期触发一次 cfg_mgmt 写到 DWADDR=1
            //     - command: rw[143:128] = 0x0007 (常见：IO/MEM/BUSMASTER)
            //     - status : rw[159:144] = 0xFF00 (这里注释写“do not update”，实际 byte_en 控制)
            //--------------------------------------------------------------------------
            if ( (rw[RWPOS_CFG_CFGSPACE_STATUS_CL_EN] | rw[RWPOS_CFG_CFGSPACE_COMMAND_EN]) &
                 ~in_cmd_read & ~in_cmd_write &
                 ~rw[RWPOS_CFG_RD_EN] & ~rw[RWPOS_CFG_WR_EN] &
                 ~rwi_cfg_mgmt_rd_en & ~rwi_cfg_mgmt_wr_en ) begin

                if (rwi_count_cfgspace_status_cl < rw[672+:32]) begin
                    rwi_count_cfgspace_status_cl <= rwi_count_cfgspace_status_cl + 1;
                end else begin
                    rwi_count_cfgspace_status_cl <= 0;

                    // 触发一次 cfg_mgmt 写
                    rw[RWPOS_CFG_WR_EN] <= 1'b1;

                    // 组装要写入的 di：低 16 -> command，高 16 -> status(但 byte_en 可能关掉)
                    rw[143:128] <= 16'h0007; // command register
                    rw[159:144] <= 16'hff00; // status register(理论上这里 byte_en 可控制不写)

                    // 访问配置空间 DWADDR=1 (即 config offset 0x04)
                    rw[169:160] <= 1;
                    rw[170]     <= 0;
                    rw[171]     <= 0;

                    // byte_en：决定写 command/status 哪些字节
                    rw[172]     <= rw[RWPOS_CFG_CFGSPACE_COMMAND_EN];   // command byte0
                    rw[173]     <= rw[RWPOS_CFG_CFGSPACE_COMMAND_EN];   // command byte1
                    rw[174]     <= 0;                                   // status byte0
                    rw[175]     <= rw[RWPOS_CFG_CFGSPACE_STATUS_CL_EN]; // status byte1
                end
            end

            //--------------------------------------------------------------------------
            // 13.4 自动读取 BAR0：
            //     当 base_address_register_reg 处于“未初始化/无效值”时触发一次 cfg_mgmt 读
            //     这里判断了 0 / FFFFC004 / 00000004 三个值为“需要重新读取”
            //--------------------------------------------------------------------------
            if ( (base_address_register_reg == 32'h00000000) ||
                 (base_address_register_reg == 32'hFFFFC004) ||
                 (base_address_register_reg == 32'h00000004) ) begin

                if ( ~in_cmd_read & ~in_cmd_write &
                     ~rw[RWPOS_CFG_RD_EN] & ~rw[RWPOS_CFG_WR_EN] &
                     ~rwi_cfg_mgmt_rd_en & ~rwi_cfg_mgmt_wr_en ) begin

                    // 触发一次 cfg_mgmt 读
                    rw[RWPOS_CFG_RD_EN] <= 1'b1;

                    // 读 DWADDR=6 (即 config offset 0x18) —— 你注释写 BAR0，
                    // 实际 BAR0 在 type0 header 是 offset 0x10(DWADDR=4)，
                    // 这里 DWADDR=6 是 BAR2/别的字段，建议你确认目标设备的 BAR 位置。
                    rw[169:160] <= 6;

                    // byte_en 全 0 (读不需要 byte_en，但这里保持原写法)
                    rw[172]     <= 0;
                    rw[173]     <= 0;
                    rw[174]     <= 0;
                    rw[175]     <= 0;
                end
            end

            //--------------------------------------------------------------------------
            // 13.5 cfg_mgmt 完成握手：保存返回数据 & 更新最近访问记录
            //--------------------------------------------------------------------------
            if (ctx.cfg_mgmt_rd_wr_done) begin
                // 若本次是“BAR 读取请求”，保存返回的 do
                if ( (base_address_register_reg == 32'h00000000) ||
                     (base_address_register_reg == 32'hFFFFC004) ||
                     (base_address_register_reg == 32'h00000004) ) begin

                    // 仅当访问 DWADDR=6 且是读使能时才保存
                    if ((ctx.cfg_mgmt_dwaddr == 8'h06) && rwi_cfg_mgmt_rd_en) begin
                        base_address_register_reg <= ctx.cfg_mgmt_do;
                    end
                end

                // 清内部 rd/wr 使能
                rwi_cfg_mgmt_rd_en <= 1'b0;
                rwi_cfg_mgmt_wr_en <= 1'b0;

                // 更新 ro 中“最近一次访问结果”
                rwi_cfgrd_valid   <= 1'b1;
                rwi_cfgrd_addr    <= ctx.cfg_mgmt_dwaddr;
                rwi_cfgrd_data    <= ctx.cfg_mgmt_do;
                rwi_cfgrd_byte_en <= ctx.cfg_mgmt_byte_en;
            end
            else if (rw[RWPOS_CFG_RD_EN]) begin
                //--------------------------------------------------------------------------
                // 13.6 将“rw 触发位”转换为内部 cfg_mgmt 读脉冲
                //--------------------------------------------------------------------------
                rw[RWPOS_CFG_RD_EN] <= 1'b0;
                rwi_cfg_mgmt_rd_en  <= 1'b1;
                rwi_cfgrd_valid     <= 1'b0;
            end
            else if (rw[RWPOS_CFG_WR_EN]) begin
                //--------------------------------------------------------------------------
                // 13.7 将“rw 触发位”转换为内部 cfg_mgmt 写脉冲
                //--------------------------------------------------------------------------
                rw[RWPOS_CFG_WR_EN] <= 1'b0;
                rwi_cfg_mgmt_wr_en  <= 1'b1;
                rwi_cfgrd_valid     <= 1'b0;
            end

            //--------------------------------------------------------------------------
            // 13.8 原始静态 TLP 状态机已被你注释掉，这里不再处理
            //--------------------------------------------------------------------------
        end
    end


    //==========================================================================
    // 14) MSI/MSI-X “cfg_interrupt_*” 逻辑(你现在基本没在用/逻辑有点怪)
    //    注意：你写的是：
    //      - ready && valid -> 拉低 valid
    //      - o_int -> cfg_int_valid <= 0 (看起来像禁用，而不是触发)
    //    如果你要用 cfg_interrupt 发 MSI/MSI-X，通常 o_int 时应拉高 valid。
    //==========================================================================
    wire o_int;

    always @ (posedge clk_pcie) begin
        if (rst) begin
            cfg_int_valid  <= 1'b0;
            cfg_msg_num    <= 5'b0;
            cfg_int_assert <= 1'b0;
            cfg_int_di     <= 8'b0;
            cfg_int_stat   <= 1'b0;
        end else if (cfg_int_ready && cfg_int_valid) begin
            cfg_int_valid <= 1'b0;
        end else if (o_int) begin
            // 你这里是 o_int 时拉低 valid（等价于“什么都不发”）
            cfg_int_valid <= 1'b0;
        end
    end


    //==========================================================================
    // 15) MSI-X TLP 注入逻辑：使用 tlps_static 通道发送 MWR64
    //    时序：
    //      - o_int && i_valid：先置 msix_has_data=1 并准备 msix_tlp
    //      - msix_has_data && tready：拉高 msix_valid 一个周期发出去
    //      - msix_valid：下一拍自动清零
    //==========================================================================
    always @ (posedge clk_pcie) begin
        if (rst) begin
            msix_valid    <= 1'b0;
            msix_has_data <= 1'b0;
            msix_tlp      <= 128'b0;
        end else if (msix_valid) begin
            // 发出后自动清 valid
            msix_valid <= 1'b0;
        end else if (msix_has_data && tlps_static.tready) begin
            // 下游 ready 时，拉高 valid 发出
            msix_valid    <= 1'b1;
            msix_has_data <= 1'b0;
        end else if (o_int && i_valid) begin
            // 触发：准备一次 TLP
            msix_has_data <= 1'b1;
            msix_tlp      <= {MWR64_DW4, MWR64_DW3, MWR64_DW2, HDR_MEMWR64};
        end
    end


    //==========================================================================
    // 16) 定时产生 o_int：int_enable=1 时计数到 100000 触发一次
    //==========================================================================
    time int_cnt = 0;
    always @ (posedge clk_pcie) begin
        if (rst) begin
            int_cnt <= 0;
        end else if (int_cnt == 32'd100000) begin
            int_cnt <= 0;
        end else if (int_enable) begin
            int_cnt <= int_cnt + 1;
        end
    end

    // 当计数到达阈值时产生一次 o_int（单拍宽度取决于 int_cnt 回绕逻辑）
    assign o_int = (int_cnt == 32'd100000);

endmodule
